<?php

namespace Drupal\commerce_credibanco;

use Drupal\Core\Url;
use Exception;

define('RBS_CREDIBANCO_PAYMENT_NAME', 'CredibanCo');

define('RBS_CREDIBANCO_TEST_URL' , 'https://ecouat.credibanco.com/payment/rest/');
define('RBS_CREDIBANCO_PROD_URL' , 'https://eco.credibanco.com/payment/rest/');

define('RBS_CREDIBANCO_TEST_URL_PROXY' , 'https://ecouat.credibanco.com/proxy/rest/');
define('RBS_CREDIBANCO_PROD_URL_PROXY' , 'https://eco.credibanco.com/proxy/rest/');


define('RBS_CREDIBANCO_ENABLE_LOGGING', true);
define('RBS_CREDIBANCO_EXT_FIELDS', false);
define('RBS_CREDIBANCO_MEASUREMENT_NAME', "p");


/**
 * Sends requests to the server encoding request parameters and decoding
 * results with given encoder and decoder. returns error if a timeout occurs
 */
class CommerceCredibanCoPaymentApi
{
    const test_API_URL = RBS_CREDIBANCO_TEST_URL;
    const production_API_URL = RBS_CREDIBANCO_PROD_URL;

    const maxFileReadingStringLength = 4096;
    const internalErrorCode = 10000;

    const orderStatusPending = 0;
    const orderStatusPreHold = 1;
    const orderStatusAuthorized = 2;
    const orderStatusReversed = 3;
    const orderStatusPartlyRefunded = 4;
    const orderStatusAuthACS = 5;
    const orderStatusDeclined = 6;

    public $cms_name = "Drupal 8.x Commerce 2";
    public $cms_version = "";
    public $module_version = '1.0.0';
    public $language = 'es';

    /**
     * @var string
     *  Well-formed server url to connect with
     */
    private $serverUrl;

    /**
     * @var int
     *  Timeout for server communications in seconds.
     */
    private $timeout;

    /**
     * @var string
     */
    private $userName;

    /**
     * @return string
     */
    public function getUserName()
    {
        return $this->userName;
    }

    /**
     * @var string
     */
    private $password;

    /**
     * @return string
     */
    public function getPassword()
    {
        return $this->password;
    }

    /**
     * @var boolean
     */
    private $isDoubleStaged;

    /**
     * @var boolean
     */
    private $isTestMode;
    private $jsonParams;

    /**
     * @var boolean
     */
    private $isLogging;

    public function __construct($params = array())
    {
        foreach ($params as $key => $value) {
            $this->{$key} = $value;
        }
    }

    function registerOrder($payment_id, $amount, $return_url, $fail_url = null, $description = "")
    {
        $jsonParams_array = array(
            'CMS:' => $this->cms_name . " " . $this->cms_version,
            'Module-Version: ' => $this->module_version
        );

        $data = array(
            'orderNumber' => $payment_id . "_" . time(),
            'amount' => $amount,
            'returnUrl' => $return_url,
            'language' => $this->language,
            'jsonParams' => json_encode(array_merge($jsonParams_array, $this->jsonParams)),
        );

        if (!empty($description)) {
            $data['description'] = $description;
        }

        if (!empty($fail_url) && $fail_url != $return_url) {
            $data['failUrl'] = $fail_url;
        }

        $method = $this->isDoubleStaged ? 'registerPreAuth.do' : 'register.do';

        return $this->callMethod($method, $data);

    }

    /**
     * Sends data compressed by deflate algorithm to the server.
     *
     * @param string $method_url_suffix
     *  Url suffix.
     * @param array $args
     *  Data to be send to server.
     *
     * @return array
     *  response.
     */
    protected function callMethod($method_url_suffix, array $args = array())
    {

        $this->serverUrl = $this->isTestMode ? self::test_API_URL : self::production_API_URL;

        try {

            $action_adr = $this->serverUrl . "/" . $method_url_suffix;

            $args['userName'] = $this->userName;
            $args['password'] = $this->password;

            $rbsCurl = curl_init();
            curl_setopt_array($rbsCurl, array(
                CURLOPT_HTTPHEADER => array(
                    'CMS: Drupal 8.x',
                    'Module-Version: ' . $this->module_version,
                ),
                CURLOPT_URL => $action_adr,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => http_build_query($args, '', '&')
            ));

            $response = curl_exec($rbsCurl);
            curl_close($rbsCurl);

            $communication_start_time = time();

            if (empty($response)) {
                $this->throwConnectionException($communication_start_time, t("Problem occurred while reading data from bank's server."));
            }

            $result = json_decode($response, TRUE);

            if ($this->isLogging) {
                if (isset($result['errorCode']) && $result['errorCode']) {
                    \Drupal::logger('commerce_credibanco')->warning('API "%payment_method" call, ErrorCode = "%error_code", message = "%message", <br/> post: %post<br/> response: %response', array(
                        '%payment_method' => $method_url_suffix,
                        '%error_code' => isset($result['errorCode']) ? $result['errorCode'] : '',
                        '%message' => isset($result['errorMessage']) ? $result['errorMessage'] : '',
                        '%post' => htmlentities(print_r($args, TRUE)),
                        '%response' => htmlentities($response),
                    ));
                } else {
                    \Drupal::logger('commerce_credibanco')->debug('API "%payment_method" call, ErrorCode = "%error_code", message = "%message", <br/> post: %post<br/> response: %response', array(
                        '%payment_method' => $method_url_suffix,
                        '%error_code' => isset($result['errorCode']) ? $result['errorCode'] : '',
                        '%message' => isset($result['errorMessage']) ? $result['errorMessage'] : '',
                        '%post' => htmlentities(print_r($args, TRUE)),
                        '%response' => htmlentities($response),
                    ));
                }
            }

            return $result;
        } catch (Exception $e) {
            $errorCode = isset($result['errorCode']) ? $result['errorCode'] : '';
            if ($errorCode) {
                \Drupal::logger('commerce_credibanco')->warning('API "%payment_method" call, ErrorCode = "%error_code", message = "%message", <br/> post: %post', array(
                    '%payment_method' => $method_url_suffix,
                    '%error_code' => $errorCode,
                    '%message' => $e->getMessage(),
                    '%post' => htmlentities(print_r($args, TRUE)),
                ));
            } else {
                \Drupal::logger('commerce_credibanco')->debug('API "%payment_method" call, ErrorCode = "%error_code", message = "%message", <br/> post: %post', array(
                    '%payment_method' => $method_url_suffix,
                    '%error_code' => $errorCode,
                    '%message' => $e->getMessage(),
                    '%post' => htmlentities(print_r($args, TRUE)),
                ));
            }

            return array(
                'errorCode' => self::internalErrorCode,
                'errorMessage' => $e->getMessage(),
            );
        }
    }

    /**
     * @param int $communication_start_time
     * @param string $message
     *
     * @throws Exception
     */
    private function throwConnectionException($communication_start_time, $message)
    {
        $timeout_msg = $this->isTimedOut($communication_start_time) ? t('Bank server response time exceeded (@timeout)', array('@timeout' => $this->timeout)) : '';
        throw new Exception($message . ' ' . $timeout_msg);
    }

    /**
     * @param int $communication_start_time
     *
     * @return bool
     */
    private function isTimedOut($communication_start_time)
    {
        return (time() - $communication_start_time) >= $this->timeout;
    }

    /**
     * @param string $rbs_order_id
     * @param int $amount
     *
     * @return mixed
     */
    public function confirmOrderPayment($rbs_order_id, $amount = 0)
    {
        return $this->callMethod('deposit.do', array(
            'orderId' => $rbs_order_id,
            'amount' => $amount,
        ));
    }

    /**
     * @param int $order_id
     *
     * @return array
     */
    public function getOrderStatusByOrderId($order_id)
    {
        return $this->callMethod('getOrderStatusExtended.do', array('orderNumber' => $order_id));
    }

    /**
     * @param string $rbs_order_id
     *
     * @return mixed
     */
    public function getOrderStatusByRBSOrderId($rbs_order_id)
    {
        return $this->callMethod('getOrderStatusExtended.do', array('orderId' => $rbs_order_id));
    }

    /**
     * @param string $rbs_order_id
     *
     * @return mixed
     */
    public function reverseOrderPayment($rbs_order_id)
    {
        return $this->callMethod('reverse.do', array('orderId' => $rbs_order_id));
    }

    /**
     * @param string $rbs_order_id
     * @param int $amount
     *
     * @return mixed
     */
    public function refundOrderPayment($rbs_order_id, $amount)
    {
        return $this->callMethod('refund.do', array(
            'orderId' => $rbs_order_id,
            'amount' => $amount
        ));
    }

    /**
     * @param string $date
     * @param array $transaction_states
     * @param bool $search_by_created_date
     * @param int $num_entries
     *
     * @return array
     */
    public function getOperationsList($date, $transaction_states = array(), $search_by_created_date = TRUE, $num_entries = 100)
    {
        if (!$transaction_states) {
            $transaction_states = array('CREATED', 'APPROVED', 'DEPOSITED', 'DECLINED', 'REVERSED', 'REFUNDED');
        }

        $params = array(
            'from' => "{$date}000000",
            'to' => "{$date}235959",
            'size' => $num_entries,
            'transactionStates' => implode(', ', $transaction_states),
            'searchByCreatedDate' => $search_by_created_date,
            'merchants' => '',
        );
        return $this->callMethod('getLastOrdersForMerchants.do', $params);
    }
}