<?php

namespace Drupal\commerce_credibanco\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\PaymentMethodTypeManager;
use Drupal\commerce_payment\PaymentTypeManager;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_price\Entity\Currency;
use Drupal\commerce_price\Price;
use Drupal\commerce_price\RounderInterface;
use Drupal\commerce_credibanco\CommerceCredibanCoPaymentApi;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Utility\Unicode;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Drupal\commerce_payment\Exception\PaymentGatewayException;

//define("MODULE_PATH", \Drupal::service('file_system')->realpath(drupal_get_path('core', '')));
//DIE(MODULE_PATH);

/**
 * Provides the Off-site CredibanCo payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "commerce_credibanco",
 *   label = @Translation("Crediabanco (Pago en linea con Credibanco)"),
 *   display_label = "CredibanCo" ,
 *   forms = {
 *     "offsite-payment" = "Drupal\commerce_credibanco\PluginForm\OffsiteRedirect\PaymentOffsiteForm",
 *   },
 *   payment_method_types = {"credit_card"},
 *   credit_card_types = {
 *     "amex", "mir", "jcb", "unionpay", "mastercard", "visa",
 *   },
 * )
 */
class CredibanCoPayment extends OffsitePaymentGatewayBase implements CredibanCoPaymentInterface
{

    /**
     * The price rounder.
     *
     * @var \Drupal\commerce_price\RounderInterface
     */
    protected $rounder;

    public function __construct(array $configuration,
                                $plugin_id,
                                $plugin_definition,
                                EntityTypeManagerInterface $entity_type_manager,
                                PaymentTypeManager $payment_type_manager,
                                PaymentMethodTypeManager $payment_method_type_manager,
                                TimeInterface $time,
                                RounderInterface $rounder)
    {
        parent::__construct($configuration, $plugin_id, $plugin_definition, $entity_type_manager, $payment_type_manager, $payment_method_type_manager, $time);
        $this->rounder = $rounder;
    }


    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {

        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('entity_type.manager'),
            $container->get('plugin.manager.commerce_payment_type'),
            $container->get('plugin.manager.commerce_payment_method_type'),
            $container->get('datetime.time'),
            $container->get('commerce_price.rounder')
        );
    }

    public function capturePayment(PaymentInterface $payment, Price $amount = NULL)
    {
        // TODO: Implement capturePayment() method.
    }

    public function refundPayment(PaymentInterface $payment, Price $amount = NULL)
    {

        // eirano
        drupal_set_message(t('Error # %code: %message', [
            '%code' => 13,
            '%message' => "Not allowed in current version of module",
        ]), 'error');
        return false;

        // TODO: Implement capturePayment() method.
    }

    /**
     * {@inheritdoc}
     */
    public function voidPayment(PaymentInterface $payment)
    {
        // TODO: Implement voidPayment() method.
    }

    /**
     * {@inheritdoc}
     */
    public function defaultConfiguration()
    {
        return [
            'username' => '',
            'password' => '',
            'double_staged' => false,
            'logging' => true,
        ] + parent::defaultConfiguration();
    }

    /**
     * {@inheritdoc}
     */
    public function buildConfigurationForm(array $form, FormStateInterface $form_state)
    {
        $form = parent::buildConfigurationForm($form, $form_state);

        $form['username'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Login-API'),
            '#default_value' => $this->configuration['username'],
            '#required' => TRUE,
        ];

        $form['password'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Password'),
            '#default_value' => $this->configuration['password'],
            '#required' => TRUE,
        ];

        $form['double_staged'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Two-phase payments'),
            '#default_value' => $this->configuration['double_staged'],
        ];

        $form['logging'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Write log-file'),
            '#default_value' => $this->configuration['logging'],
        ];

        return $form;
    }

    /**
     * {@inheritdoc}
     */
    public function submitConfigurationForm(array &$form, FormStateInterface $form_state)
    {
        parent::submitConfigurationForm($form, $form_state);
        if (!$form_state->getErrors()) {
            $values = $form_state->getValue($form['#parents']);

            $this->configuration['username'] = $values['username'];

            if (!empty($values['password'])) {
                $this->configuration['password'] = $values['password'];
            }
            $this->configuration['double_staged'] = $values['double_staged'];
            $this->configuration['logging'] = $values['logging'];

            $module_handler = \Drupal::service('module_handler');
            $module_path = $module_handler->getModule('commerce_credibanco')->getPath();
            $img_logo = " <img src=\"" . $GLOBALS['base_url'] . "/" . $module_path ."/img/logo.png\" />" ;
            $this->configuration['display_label'] = $values['display_label'] . $img_logo;
        }
//        print_r(get_class_methods($this));die;
    }

    /**
     * {@inheritdoc}
     */
    public function onReturn(OrderInterface $order, Request $request)
    {

        // and so, why we get 403 on callback
        // https://www.drupal.org/project/commerce/issues/3051241

        $callback = $request->query->get('callback');
        if (isset($callback)) {
            $remote_id = $request->query->get('mdOrder');
        } else {
            $remote_id = $request->query->get('orderId');
        }

        if (!isset($remote_id)) {
            die("Failed to request!");
        }

        // Set REST API url for test or live modes.
        switch ($this->getMode()) {
            default:
            case 'test':
                $action_adr = CommerceCredibanCoPaymentApi::test_API_URL;
                break;

            case 'live':
                $action_adr = CommerceCredibanCoPaymentApi::production_API_URL;
                break;
        }

        $action_adr .= 'getOrderStatusExtended.do';

        $args = array(
            'userName' => $this->configuration['username'],
            'password' => $this->configuration['password'],
            'orderId' => $remote_id
        );

        $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');

        $rbsCurl = curl_init();
        curl_setopt_array($rbsCurl, array(
            CURLOPT_URL => $action_adr,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_POSTFIELDS => http_build_query($args, '', '&'),
        ));

        $response = curl_exec($rbsCurl);
        curl_close($rbsCurl);

        $response = json_decode($response, true);

        $payment = $payment_storage->loadByRemoteId($remote_id);

        $orderStatus = $response['orderStatus'];
        if ($orderStatus == '1' || $orderStatus == '2') {

            $payment->setState('completed');
            $payment->setAmount($order->getTotalPrice());
            $payment->setRemoteState("Payed");
            $payment->setCompletedTime(time());
            $payment->save();

        } else {
            $payment->setState('authorization_voided');
            $payment->save();
            throw new PaymentGatewayException('Payment failed!');
        }

        $this->setLocalState($payment, $orderStatus);

    }

    /**
     * Sets transaction 'status' and 'message' depending on RBS status.
     *
     * @param PaymentInterface $payment
     * @param int $remote_status
     */
    public function setLocalState(PaymentInterface $payment, $remote_status)
    {
        switch ($remote_status) {
            case CommerceCredibanCoPaymentApi::orderStatusPending:
                $payment->setState('pending');
                break;
            case CommerceCredibanCoPaymentApi::orderStatusPreHold:
                $payment->setState('authorization');
                break;

            case CommerceCredibanCoPaymentApi::orderStatusAuthorized:
                $payment->setState('completed');
                break;

            case CommerceCredibanCoPaymentApi::orderStatusDeclined:
                $payment->setState('authorization_voided');
                break;

            case CommerceCredibanCoPaymentApi::orderStatusPartlyRefunded:
                $status = $payment->getBalance()->isZero() ? 'refunded' : 'partially_refunded';
                $payment->setState($status);
                break;

            case CommerceCredibanCoPaymentApi::orderStatusReversed:
                $payment->setState('refunded');
                break;
        }
    }

    /**
     * {@inheritdoc}
     */
    public function onCancel(OrderInterface $order, Request $request)
    {
        parent::onCancel($order, $request);
    }

    /**
     * {@inheritdoc}
     */
    public function onNotify(Request $request)
    {

        $data = $request->query->all();

        /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
        $payment = $this->entityTypeManager->getStorage('commerce_payment')->load($data['orderNumber']);
        if (is_null($payment) || $payment->getRemoteId() != $data['mdOrder']) {
            return FALSE;
        }

        $status_response = $this->getApi($payment)->getOrderStatusByRBSOrderId($data['mdOrder']);
        switch ($data['operation']) {
            case 'approved':
                break;
            case 'deposited':
                $status = $this->doDeposited($payment, $status_response);
                break;
            case 'reversed':
            case 'refunded':
                $status = $this->doRefunded($payment, $status_response);
                break;
            case 'declinedByTimeout':
                $status = $this->doCancel($payment, $status_response);
                break;
        }

        if (!$status) {
            return FALSE;
        }

        $payment->save();
        $payment->getOrder()->setData($this->getPluginId(), $status_response)->save();
    }

    public function getApi(PaymentInterface $payment)
    {
        /** @var \Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayInterface $payment_gateway_plugin */
        $payment_gateway_plugin = $payment->getPaymentGateway()->getPlugin();

        $payment_gateway_configuration = $payment_gateway_plugin->getConfiguration();

        $params['userName'] = $payment_gateway_configuration['username'];
        $params['password'] = $payment_gateway_configuration['password'];
        $params['isDoubleStaged'] = $payment_gateway_configuration['double_staged'] == 0 ? false : true;
        $params['isTestMode'] = $payment->getPaymentGatewayMode() == 'live' ? false : true;
        $params['isLogging'] = $payment_gateway_configuration['logging'] == 0 ? false : true;

        return new CommerceCredibanCoPaymentApi($params);
    }

    protected function doDeposited(PaymentInterface $payment, array $status_response)
    {
        $response_amount = intdiv($status_response['amount'], 100);
        $gateway_price = new Price($response_amount, $this->getCurrencyCode($status_response['currency']));
        if (!$payment->getAmount()->equals($gateway_price)) {
            return FALSE;
        }
        $payment->setRemoteState($status_response['paymentAmountInfo']['paymentState']);
        $this->setLocalState($payment, $status_response['orderStatus']);

        return TRUE;
    }

    /**
     * @param string $numeric_currency_code
     *
     * @return string
     */
    function getCurrencyCode($numeric_currency_code)
    {
        /** @var Currency[] $currencies */
        $currencies = Currency::loadMultiple();
        foreach ($currencies as $currency) {
            if ($currency->getNumericCode() == $numeric_currency_code) {
                return $currency->getCurrencyCode();
            }
        }
        return NULL;
    }

    protected function doRefunded(PaymentInterface $payment, array $status_response)
    {
        //@todo need additional checks here ?
        $response_amount = intdiv($status_response['paymentAmountInfo']['refundedAmount'], 100);
        $refunded_amount = new Price($response_amount, $this->getCurrencyCode($status_response['currency']));
        $payment->setRefundedAmount($refunded_amount);
        $payment->setRemoteState($status_response['paymentAmountInfo']['paymentState']);
        $this->setLocalState($payment, $status_response['orderStatus']);

        return TRUE;
    }

    protected function doCancel(PaymentInterface $payment, array $status_response)
    {
        $payment->setRemoteState($status_response['paymentAmountInfo']['paymentState']);
        $payment->setState('authorization_expired');

        return TRUE;
    }

    /**
     * {@inheritdoc}
     */
    public function buildPaymentOperations(PaymentInterface $payment)
    {
        $payment_state = $payment->getState()->value;
        $operations = parent::buildPaymentOperations($payment);
        // @todo Grant reversal access to single staged payment.

        return $operations;
    }
}
