<?php

namespace RBS\CredibanCo\Model;

use Magento\Quote\Api\Data\CartInterface;
use Magento\Sales\Model\Order\Payment\Transaction;
use Magento\Sales\Model\Order;

require_once(__DIR__ . '/Config/include.php');

/**
 * Class CredibanCo
 * @package RBS\CredibanCo\Model
 */
class CredibanCo extends \Magento\Payment\Model\Method\AbstractMethod
{

    protected $RBS_CREDIBANCO_PROD_URL = RBS_CREDIBANCO_PROD_URL;
    protected $RBS_CREDIBANCO_TEST_URL = RBS_CREDIBANCO_TEST_URL;

    /**
     * @var bool
     */
    protected $_isInitializeNeeded = true;
    protected $_isGateway = true;
    /**
     * Payment code
     *
     * @var string
     */
    protected $_code = 'credibanco';

    /**
     * Availability option
     *
     * @var bool
     */
    protected $_isOffline = false;
    /**
     * Sidebar payment info block
     *
     * @var string
     */
    protected $_infoBlockType = 'Magento\Payment\Block\Info\Instructions';

    protected $_encryptor;

    protected $orderFactory;

    protected $urlBuilder;

    protected $_transactionBuilder;

    protected $_logger;

    public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Payment\Model\Method\Logger $logger,
        \Magento\Framework\Module\ModuleListInterface $moduleList,
        \Magento\Framework\Stdlib\DateTime\TimezoneInterface $localeDate,
        \Magento\Framework\Encryption\EncryptorInterface $encryptor,
        \Magento\Framework\UrlInterface $urlBuilder,
        \Magento\Sales\Model\Order\Payment\Transaction\BuilderInterface $builderInterface,
        \Magento\Sales\Model\OrderFactory $orderFactory,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        array $data = [])
    {
        $this->orderFactory = $orderFactory;
        $this->urlBuilder = $urlBuilder;
        $this->_transactionBuilder = $builderInterface;
        $this->_encryptor = $encryptor;
        parent::__construct($context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $logger,
            $resource,
            $resourceCollection,
            $data);
//        $writer = new \Zend\Log\Writer\Stream(BP . '/var/log/CredibanCo.log');
//        $this->_logger = new \Zend\Log\Logger();
//        $this->_logger->addWriter($writer);
    }

    /**
     *
     * @param $orderId
     * @return int|null
     */
    public function getCustomerId($orderId)
    {
        return $this->getOrder($orderId)->getCustomerId();
    }

    /**
     *
     * @param $orderId
     * @return Order
     */
    protected function getOrder($orderId)
    {
        return $this->orderFactory->create()->loadByIncrementId($orderId);
    }

    /**
     *
     * @param $orderId
     * @return null|string
     */
    public function getCurrencyCode($orderId)
    {
        return $this->getOrder($orderId)->getBaseCurrencyCode();
    }

    /**
     * Set order state and status
     *
     * @param string $paymentAction
     * @param \Magento\Framework\DataObject $stateObject
     * @return void
     */
    public function initialize($paymentAction, $stateObject)
    {
        $stateObject->setState(Order::STATE_PENDING_PAYMENT);
        $stateObject->setStatus(Order::STATE_PENDING_PAYMENT);
        $stateObject->setIsNotified(false);
    }

    /**
     * Check whether payment method can be used
     *
     * @param CartInterface|null $quote
     * @return bool
     */
    public function isAvailable(\Magento\Quote\Api\Data\CartInterface $quote = null)
    {
        if ($quote === null) {
            return false;
        }
        return parent::isAvailable($quote) && $this->isCarrierAllowed(
            $quote->getShippingAddress()->getShippingMethod()
        );
    }

    /**
     * Check whether payment method can be used with selected shipping method
     *
     * @param string $shippingMethod
     * @return bool
     */
    protected function isCarrierAllowed($shippingMethod)
    {
        return strpos($this->getConfigData('allowed_carrier'), $shippingMethod) !== false;
    }

    /**
     *
     * @param bool $with_action
     * @return string
     */
    public function getGateUrl($with_action = true)
    {
        $action_adr = $this->RBS_CREDIBANCO_PROD_URL;
        if ($this->getConfigData('test_mode')) {
            $action_adr = $this->RBS_CREDIBANCO_TEST_URL;
        }

        if ($with_action == false) {
            return $action_adr;
        }

        if ($this->getConfigData('two_stage')) {
            $action_adr .= "registerPreAuth.do";
        } else {
            $action_adr .= "register.do";
        }

        return $action_adr;
    }

    /**
     * Получить массив параметров для формы оплаты
     *
     * @param $orderId
     * @return array
     */
    public function getPostData($orderId)
    {
        $postData = array(
            'order_id' => $orderId,
            'amount' => round(number_format($this->getAmount($orderId), 2, '.', '') * 100),
        );

        return $postData;
    }

    /**
     *
     * @param $orderId
     * @return float
     */
    public function getAmount($orderId)
    {
        return $this->getOrder($orderId)->getGrandTotal();
    }

    /**
     * @param $responseData
     * @return Order
     */
    public function processResponse($responseData, $callback = false)
    {

        $debugData = ['response' => $responseData];
//        $this->_logger->debug("processResponse", $debugData);

//        if ($this->checkRBSResponse($responseData)) {

            $action_adr = $this->getStatusUrl();

            $orderId = $callback == true ? $responseData['mdOrder'] : $responseData['orderId']; //bzz

            $args = array(
                'userName' => $this->getConfigData("RBS_MERCHANT_LOGIN"),
                'password' => $this->_encryptor->decrypt($this->getConfigData("RBS_MERCHANT_PASSWORD")),
                'orderId' => $orderId
            );

            $rbsCurl = curl_init();
            curl_setopt_array($rbsCurl, array(
                CURLOPT_URL => $action_adr,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => http_build_query($args, '', '&')
            ));

            $response = curl_exec($rbsCurl);
            curl_close($rbsCurl);

            $response = json_decode($response, true);

            list($orderId,) = explode('#', $response['orderNumber']);
            $order = $this->getOrder($orderId);
            if ($order && ($this->_processOrder($order, $response, $callback) === true)) {
                // echo __("OK");
                return true;
            } else {
                // echo __("FAIL");
                return false;
            }
//        }
    }

    /**
     *
     * @param $response
     * @return bool
     */
    private function checkRBSResponse($response)
    {
//        $this->_logger->debug("checkRBSResponse");
        foreach (
            [
                "orderId",
            ] as $param) {
            if (!isset($response[$param])) {
//                $this->_logger->debug("Pay URL: required field \"{$param}\" is missing");
                return false;
            }
        }
        $settings = array(
            'login' => $this->getConfigData("RBS_MERCHANT_LOGIN"),
            'password' => $this->_encryptor->decrypt($this->getConfigData("RBS_MERCHANT_PASSWORD"))
        );

        $validated = $this->isPaymentValid($settings, $response);
        if ($validated === true) {
//            $this->_logger->debug("Response - OK");
            return true;
        } else {
//            $this->_logger->debug($validated);
            return false;
        }

    }

    public function isPaymentValid($rbsSettings, $response)
    {


//
//        'userName' => $this->getConfigData("RBS_MERCHANT_LOGIN"),
//        'password' => $this->_encryptor->decrypt($this->getConfigData("RBS_MERCHANT_PASSWORD")),
//

//
//        if ($rbsSettings['merchant_id'] != $response['merchant_id']) {
//            return 'An error has occurred during payment. Merchant data is incorrect.';
//        }
//        if ($response['order_status'] == 'declined') {
//            return 'An error has occurred during payment. Order is declined.';
//        }
//
//        $responseSignature = $response['signature'];
//        if (isset($response['response_signature_string'])) {
//            unset($response['response_signature_string']);
//        }
//        if (isset($response['signature'])) {
//            unset($response['signature']);
//        }
//        if ($this->getSignature($response, $rbsSettings['secret_key']) != $responseSignature) {
//            return 'An error has occurred during payment. Signature is not valid.';
//        }


        return true;
    }

    public function getStatusUrl()
    {
        $action_adr = $this->RBS_CREDIBANCO_PROD_URL;

        if ($this->getConfigData('test_mode')) {
            $action_adr = $this->RBS_CREDIBANCO_TEST_URL;
        }

        $action_adr .= "getOrderStatusExtended.do";

        return $action_adr;
    }

    /**
     *
     * @param Order $order
     * @param mixed $response
     * @return bool
     */
    protected function _processOrder(Order $order, $response, $callback = false)
    {

//        $this->_logger->debug("_process_CredibanCo",
//            [
//                "\$order" => $order,
//                "\$response" => $response
//            ]);
        try {
            if (round($order->getGrandTotal() * 100) != $response["amount"]) {
//                $this->_logger->debug("_processOrder: amount mismatch, order FAILED");
                return false;
            }
            if ($response["orderStatus"] == '2' || $response["orderStatus"] == '1') {

                if ($callback) {
                    //todo EIRAno
//                $this->createTransaction($order, $response);
                    $order
                        ->setState($this->getConfigData("order_status"))
                        ->setStatus($order->getConfig()->getStateDefaultStatus($this->getConfigData("order_status")))
                        ->save();
//                    $this->_logger->debug("_processOrder: order state changed: STATE_PROCESSING");
//                    $this->_logger->debug("_processOrder: order data saved, order OK");
                }

                return true;

            } else {

                if ($callback) {
                    $order
                        ->setState(Order::STATE_CANCELED)
                        ->setStatus($order->getConfig()->getStateDefaultStatus(Order::STATE_CANCELED))
                        ->save();
//                    $this->_logger->debug("_processOrder: order state not STATE_CANCELED");
//                    $this->_logger->debug("_processOrder: order data saved, order not approved");
                }

                return false;
            }

        } catch (\Exception $e) {
//            $this->_logger->debug("_processOrder exception", $e->getTrace());
            return false;
        }
    }


    public function createTransaction($order = null, $paymentData = array())
    {
        try {
            //get payment object from order object
            $payment = $order->getPayment();

            $payment->setLastTransId($paymentData['payment_id']);
            $payment->setTransactionId($paymentData['payment_id']);
            $payment->setAdditionalInformation(
                [\Magento\Sales\Model\Order\Payment\Transaction::RAW_DETAILS => (array)$paymentData]
            );
            $formatedPrice = $order->getBaseCurrency()->formatTxt(
                $order->getGrandTotal()
            );

            $message = __('The authorized amount is %1.', $formatedPrice);
            //get the object of builder class
            $trans = $this->_transactionBuilder;
            $transaction = $trans->setPayment($payment)
                ->setOrder($order)
                ->setTransactionId($paymentData['payment_id'])
                ->setAdditionalInformation(
                    [\Magento\Sales\Model\Order\Payment\Transaction::RAW_DETAILS => (array)$paymentData]
                )
                ->setFailSafe(true)
                //build method creates the transaction and returns the object
                ->build(\Magento\Sales\Model\Order\Payment\Transaction::TYPE_ORDER);

            $payment->addTransactionCommentsToOrder(
                $transaction,
                $message
            );
            $payment->setParentTransactionId(null);
            $payment->save();
            $order->save();

            return $transaction->save()->getTransactionId();
        } catch (\Exception $e) {
//            $this->_logger->debug("_processOrder exception", $e->getTrace());
        }
    }
}